<?php
/**
 * Plugin Name: E-Invoicing Connector for Scrada Lite
 * Plugin URI:  https://wpmajestic.com/
 * Description: Automatically sends B2B orders to Scrada for Peppol/E-invoicing compliance (Belgium 2026). Requires: WooCommerce and a supported VAT plugin (e.g. WooCommerce EU VAT Compliance).
 * Version:     1.0
 * Author:      wpmajestic
 * Author URI:  https://wpmajestic.com/
 * Text Domain: e-invoicing-connector-for-scrada-lite
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// HPOS Compatibility
add_action( 'before_woocommerce_init', function() {
	if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
	}
} );

// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals
class E_Invoicing_Connector_For_Scrada_Lite {

	private $api_test_url = 'https://apitest.scrada.be/v1';
	private $api_prod_url = 'https://api.scrada.be/v1';

	public function __construct() {
		// PRO DOMINANCE: If Pro is active, do nothing (yield to Pro)
		if ( class_exists( 'E_Invoicing_Connector_For_Scrada' ) ) {
			add_action( 'admin_notices', function() {
				echo '<div class="notice notice-info"><p><strong>Scrada Lite</strong> is effectively disabled because <strong>Scrada Pro</strong> is active.</p></div>';
			});
			return;
		}

		// Add Settings Menu
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		
		// Hook into Order Completion (or Processing)
		// Using 'completed' ensures the order is final before sending an official invoice
		// LITE VERSION: No Automation
		// add_action( 'woocommerce_order_status_completed', array( $this, 'send_invoice_to_scrada' ), 10, 1 );

		// Add Manual Meta Box for Lite Version
		add_action( 'add_meta_boxes', array( $this, 'add_scrada_meta_box' ) );
		add_action( 'save_post', array( $this, 'process_manual_send_to_scrada' ) );

		// Add Settings link to plugins page
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'add_settings_link' ) );
		
		// AJAX for Testing Connection
		add_action( 'wp_ajax_scrada_verify_connection', array( $this, 'ajax_verify_connection' ) );

        // Enqueue Admin Scripts
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
	}

	// --- 1. SETTINGS PAGE ---
	public function add_settings_link( $links ) {

		$settings_link = '<a href="options-general.php?page=scrada_connector">Settings</a>';
		$pro_link      = '<a href="#" target="_blank" class="scrada-go-pro-link">Go Pro</a>';
		array_unshift( $links, $settings_link, $pro_link );
		return $links;
	}

	public function add_admin_menu() {
		add_options_page(
			'Scrada Settings',
			'Scrada Connector',
			'manage_options',
			'scrada_connector',
			array( $this, 'settings_page_html' )
		);
	}

	public function register_settings() {
		register_setting( 'scrada_options', 'scrada_api_key', array( 'sanitize_callback' => 'sanitize_text_field' ) );
		register_setting( 'scrada_options', 'scrada_api_secret', array( 'sanitize_callback' => 'sanitize_text_field' ) );
		register_setting( 'scrada_options', 'scrada_company_id', array( 'sanitize_callback' => 'sanitize_text_field' ) );
		register_setting( 'scrada_options', 'scrada_journal_code', array( 'sanitize_callback' => 'sanitize_text_field' ) );
		register_setting( 'scrada_options', 'scrada_bank_iban', array( 'sanitize_callback' => 'sanitize_text_field' ) );
		register_setting( 'scrada_options', 'scrada_bank_bic', array( 'sanitize_callback' => 'sanitize_text_field' ) );
		register_setting( 'scrada_options', 'scrada_test_mode', array( 'sanitize_callback' => 'intval' ) );
		register_setting( 'scrada_options', 'scrada_debug_mode', array( 'sanitize_callback' => 'intval' ) );
	}

    public function enqueue_admin_scripts( $hook ) {
        // Enqueue on Settings Page OR Shop Order page (for manual send metabox)
        $screen = get_current_screen();
        if ( 'settings_page_scrada_connector' === $hook || ( $screen && 'shop_order' === $screen->post_type ) ) {
             wp_enqueue_style( 'scrada-lite-admin-css', plugin_dir_url( __FILE__ ) . 'assets/css/admin.css', array(), '1.0' );
        }

        if ( 'settings_page_scrada_connector' === $hook ) {
            wp_enqueue_script( 
                'scrada-lite-admin-js', 
                plugin_dir_url( __FILE__ ) . 'assets/js/admin.js', 
                array( 'jquery' ), 
                '1.0', 
                true 
            );

            wp_localize_script( 'scrada-lite-admin-js', 'scradaLiteAdminParams', array(
                'nonce' => wp_create_nonce( 'scrada_verify_nonce' )
            ));
        }
    }

	public function settings_page_html() {
		?>
		<div class="wrap">
			<h1>E-Invoicing Connector for Scrada</h1>
			<p>Connect your WooCommerce store to Scrada for automatic Peppol E-Invoicing compliance.</p>
			
			<div class="notice notice-info inline">
				<p><strong>Important:</strong> This plugin requires a supported VAT plugin (e.g. WooCommerce EU VAT Compliance).</p>
				<p>Please ensure you have checked <strong>"Show field for base country"</strong> in the <strong>WooCommerce Settings page for EU VAT Compliance</strong> (or your equivalent VAT plugin settings).</p>
			</div>

			<!-- UPSELL BANNER -->
			<div class="notice notice-success inline scrada-upsell-banner">
				<h3>🚀 Upgrade to Pro for Full Automation</h3>
				<p>You are using the <strong>Lite Version</strong> (Manual Sync). Save time and automate your invoicing!</p>
				<ul>
					<li><strong>Automatic Sync:</strong> Invoices sent instantly when order is Completed.</li>

					<li><strong>Bulk Actions:</strong> Send multiple invoices at once.</li>
					<li><strong>Priority Support:</strong> Direct access to developers.</li>
				</ul>
				<p class="button-wrapper">
					<!-- Replace # with your actual buy link -->
					<a href="#" target="_blank" class="button button-primary">Get Pro Version (€59/year)</a>
				</p>
			</div>

			<form method="post" action="options.php">
				<?php settings_fields( 'scrada_options' ); ?>
				<table class="form-table">
					<tr valign="top">
						<th scope="row">API Key (X-API-KEY)</th>
						<td><input type="text" name="scrada_api_key" value="<?php echo esc_attr( get_option( 'scrada_api_key' ) ); ?>" style="width: 400px;" /></td>
					</tr>
					<tr valign="top">
						<th scope="row">API Secret (X-PASSWORD)</th>
						<td><input type="password" name="scrada_api_secret" value="<?php echo esc_attr( get_option( 'scrada_api_secret' ) ); ?>" style="width: 400px;" /></td>
					</tr>
					<tr valign="top">
						<th scope="row">Company ID</th>
						<td><input type="text" name="scrada_company_id" value="<?php echo esc_attr( get_option( 'scrada_company_id' ) ); ?>" /></td>
					</tr>
					<tr valign="top">
						<th scope="row">Journal Code</th>
						<td>
							<input type="text" name="scrada_journal_code" value="<?php echo esc_attr( get_option( 'scrada_journal_code', 'V1' ) ); ?>" class="regular-text" />
							<p class="description">e.g. <code>V1</code>, <code>WEB</code>. Check your Scrada settings.</p>
						</td>
					</tr>
					<tr valign="top">
						<th scope="row">Bank IBAN</th>
						<td>
							<input type="text" name="scrada_bank_iban" value="<?php echo esc_attr( get_option( 'scrada_bank_iban' ) ); ?>" class="regular-text" placeholder="BE00 0000 0000 0000"/>
							<p class="description">Required for Wire Transfer instructions on unpaid invoices.</p>
						</td>
					</tr>
					<tr valign="top">
						<th scope="row">Bank BIC</th>
						<td>
							<input type="text" name="scrada_bank_bic" value="<?php echo esc_attr( get_option( 'scrada_bank_bic' ) ); ?>" class="regular-text" />
						</td>
					</tr>

					<tr valign="top">
						<th scope="row">Test Mode</th>
						<td>
							<input type="checkbox" name="scrada_test_mode" value="1" <?php checked( 1, get_option( 'scrada_test_mode' ), true ); ?> />
							<label>Send to <code>apitest.scrada.be</code> (Sandbox)</label>
						</td>
					</tr>
					<tr valign="top">
						<th scope="row">Debug Mode</th>
						<td>
							<input type="checkbox" name="scrada_debug_mode" value="1" <?php checked( 1, get_option( 'scrada_debug_mode' ), false ); ?> />
							<label>Log API payloads to <code>debug.log</code></label>
						</td>
					</tr>
				</table>
				
				<p>
					<button type="button" id="scrada_test_connection" class="button button-secondary">Test Connection</button>
					<span id="scrada_connection_status" style="margin-left: 10px; font-weight: bold;"></span>
				</p>

				<?php submit_button(); ?>
			</form>
			
			<script>
			// Moved to assets/js/admin.js
			</script>
		</div>
		<?php
	}

	// --- 2. MAIN LOGIC ---
	public function send_invoice_to_scrada( $order_id ) {
		// LIMIT CHECK for Lite Version
		if ( ! $this->check_and_increment_usage() ) {
			// If called manually (which it always is in Lite), we need to warn user.
			// Since this hook fires on save_post/AJAX, we typically add an order note or use the return value.
			$order = wc_get_order( $order_id );
			if ( $order ) {
				$order->add_order_note( 'Scrada Lite Limit Reached (10 Invoices/Month). Invoice NOT sent. Please Upgrade to Pro.' );
			}
			return;
		}

		$order = wc_get_order( $order_id );
		
		if ( ! $order ) {
			return;
		}

		// Prevent duplicates: Check if we already sent this
		if ( $order->get_meta( '_scrada_invoice_sent' ) ) {
			return;
		}

		// --- UNIVERSAL VAT FINDER ---
		// Different plugins save the VAT number in different meta keys.
		// We check them all to ensure compatibility.
		$possible_keys = array(
			'_vat_number',          // Simba / Aelia
			'_billing_vat',         // Official WooCommerce EU VAT
			'_billing_vat_number',  // Common alternative
			'vat_number',           // Generic
			'VAT Number',           // Found in user export (Case sensitive/Space)
			'_eu_vat_number',       // Booster / Other
		);

		$vat_number = '';
		foreach ( $possible_keys as $key ) {
			$val = $order->get_meta( $key );
			if ( ! empty( $val ) ) {
				$vat_number = $val;
				break; // Found it, stop looking
			}
		}
		
		// LOGIC: Only send if there is a VAT number (B2B)
		if ( empty( $vat_number ) ) {
			return; 
		}

		// Clean the VAT number (remove dots/spaces/dashes)
		$vat_number_clean = preg_replace( '/[^A-Z0-9]/', '', strtoupper( $vat_number ) );

		// --- DYNAMIC LINE ITEM LOOP & TOTALS CALCULATION ---
		$lines = array();
		$vat_groups = array(); // To track totals per rate for vatTotals
		$line_number_count = 1;
		
		foreach ( $order->get_items() as $item ) {
			$product = $item->get_product();
			
			// Calculate VAT Percentage dynamically
			$line_total = (float) $item->get_total(); // Net total (excl tax)
			$line_tax   = (float) $item->get_total_tax();
			
			$vat_rate = 0;
			if ( $line_total > 0 ) {
				$vat_rate = round( ( $line_tax / $line_total ) * 100 );
			} elseif ( $product && 'taxable' === $product->get_tax_status() ) {
				$vat_rate = 0; 
			}

			// Normalize VAT to standard Belgian rates (0, 6, 12, 21)
			$vat_rate = $this->normalize_belgian_vat( $vat_rate );

			// Scrada API requires specific fields for amounts.
			$unit_price_ex_vat = (float) $order->get_item_subtotal( $item, false, true );
			$line_excl_vat     = $unit_price_ex_vat * $item->get_quantity();
			
			// Logic for vatType: 1=Standard (21,12,6), 2=Zero/Exempt (0)
			// This prevents Error 110387 "Standard VAT rate requires > 0%"
			$vat_type = ( $vat_rate > 0 ) ? 1 : 2;

			$lines[] = array(
				'lineNumber'    => (string) $line_number_count, // Force String for Scrada
				'itemName'      => $item->get_name(),
				'quantity'      => $item->get_quantity(),
				'ItemExclVat'   => number_format( $unit_price_ex_vat, 4, '.', '' ), // 4 decimals for precision
				'TotalExclVat'  => number_format( $line_excl_vat, 2, '.', '' ),
				'vatPercentage' => $vat_rate,
				'vatType'       => $vat_type
			);
			$line_number_count++;

			// Accumulate for vatTotals (Summing ROUNDED values)
			$line_excl_rounded = round( $line_excl_vat, 2 );
			$line_tax_rounded  = round( $line_tax, 2 );

			$vat_key = (string) $vat_rate;
			if ( ! isset( $vat_groups[ $vat_key ] ) ) {
				$vat_groups[ $vat_key ] = array( 'excl' => 0, 'vat' => 0 );
			}
			$vat_groups[ $vat_key ]['excl'] += $line_excl_rounded;
			$vat_groups[ $vat_key ]['vat']  += $line_tax_rounded;
		}

		// --- SHIPPING LOOP ---
		if ( $order->get_shipping_total() > 0 ) {
			$shipping_total = (float) $order->get_shipping_total();
			$shipping_tax   = (float) $order->get_shipping_tax();
			
			$shipping_rate = 0;
			if ( $shipping_total > 0 ) {
				$shipping_rate = round( ( $shipping_tax / $shipping_total ) * 100 );
			}
			$shipping_rate = $this->normalize_belgian_vat( $shipping_rate );
			$vat_type = ( $shipping_rate > 0 ) ? 1 : 2;

			$lines[] = array(
				'lineNumber'    => (string) $line_number_count, // Force String
				'itemName'      => 'Shipping: ' . $order->get_shipping_method(),
				'quantity'      => 1,
				'ItemExclVat'   => round( $shipping_total, 4 ),
				'TotalExclVat'  => round( $shipping_total, 2 ), 
				'vatPercentage' => $shipping_rate,
				'vatType'       => $vat_type
			);
			$line_number_count++;

			// Accumulate Shipping for vatTotals (Rounded)
			$shipping_total_rounded = round( $shipping_total, 2 );
			$shipping_tax_rounded   = round( $shipping_tax, 2 );

			$vat_key = (string) $shipping_rate;
			if ( ! isset( $vat_groups[ $vat_key ] ) ) {
				$vat_groups[ $vat_key ] = array( 'excl' => 0, 'vat' => 0 );
			}
			$vat_groups[ $vat_key ]['excl'] += $shipping_total_rounded;
			$vat_groups[ $vat_key ]['vat']  += $shipping_tax_rounded;
		}

		// --- BUILD VAT TOTALS ARRAY & CALCULATE PAYLOAD TOTALS ---
		$vat_totals_api = array();
		
		$calc_total_excl = 0.0;
		$calc_total_vat  = 0.0;

		foreach ( $vat_groups as $rate => $amounts ) {
			$v_rate = (float) $rate;
			
			// $amounts are already sums of rounded items
			$group_excl = round( $amounts['excl'], 2 );
			$group_vat  = round( $amounts['vat'], 2 );
			$group_incl = $group_excl + $group_vat;

			$vat_totals_api[] = array(
				'vatPercentage' => $v_rate,
				'totalExclVat'  => number_format( $group_excl, 2, '.', '' ),
				'totalVat'      => number_format( $group_vat, 2, '.', '' ),
				'totalInclVat'  => number_format( $group_incl, 2, '.', '' ),
				'vatType'       => ( $v_rate > 0 ) ? 1 : 2
			);

			$calc_total_excl += $group_excl;
			$calc_total_vat  += $group_vat;
		}

		$calc_total_incl_vat = $calc_total_excl + $calc_total_vat;

		// --- 4. PREPARE PAYLOAD ---
		$book_year = gmdate( 'Y', strtotime( $order->get_date_created() ) );
		$journal   = get_option( 'scrada_journal_code', 'V1' ); // Default to V1 as per user env
		
		// Use CALCULATED totals
		$total_incl_vat = $calc_total_incl_vat;
		
		$total_excl_vat = $calc_total_excl; // for logic below

		// Payment Logic
		$is_paid = $order->is_paid();
		$total_paid = $is_paid ? $total_incl_vat : 0;
		$total_to_pay = $is_paid ? 0 : $total_incl_vat;

		$payment_methods = array();
		
		// Use Type 1 (Wire Transfer) as default/fallback
		$payment_method_entry = array(
			'paymentType'      => 1, // 1 = Wire Transfer / Transfer
			'name'             => 'Wire Transfer',
			'paymentReference' => (string) $order->get_order_number(),
			'totalPaid'        => number_format( $total_paid, 2, '.', '' ),
			'totalToPay'       => number_format( $total_to_pay, 2, '.', '' )
		);

		// Only add IBAN/BIC if we have them
		$iban = get_option( 'scrada_bank_iban' );
		$bic  = get_option( 'scrada_bank_bic' );
		
		// Sanitize IBAN
		if ( ! empty( $iban ) ) {
			$iban_clean = preg_replace( '/[^A-Z0-9]/', '', strtoupper( $iban ) );
			$payment_method_entry['iban'] = $iban_clean;
		}
		if ( ! empty( $bic ) ) {
			$payment_method_entry['bic'] = sanitize_text_field( $bic );
		}

		$payment_methods[] = $payment_method_entry;

		// Attachments (PDF) - Ported from Pro
		$attachments = array();
		if ( function_exists( 'wcpdf_get_document' ) ) {
			try {
				$pdf_document = wcpdf_get_document( 'invoice', $order );
				if ( $pdf_document ) {
					$pdf_data = $pdf_document->get_pdf();
					if ( $pdf_data ) {
						$attachments[] = array(
							'filename'   => 'invoice_' . $order->get_order_number() . '.pdf',
							'fileType'   => 1,
							'mimeType'   => 'application/pdf',
							'base64Data' => base64_encode( $pdf_data )
						);
					}
				}
			} catch ( Exception $e ) {
				if ( get_option( 'scrada_debug_mode' ) ) {
					error_log( 'Scrada PDF Generation Error: ' . $e->getMessage() );
				}
			}
		}

		$payload = array(
			'bookYear'      => $book_year,
			'journal'       => $journal, 
			'number'        => (string) $order->get_order_number(),
			'invoiceDate'   => gmdate( 'Y-m-d', strtotime( $order->get_date_created() ) ),
			'customer'      => array(
				'name'      => $order->get_billing_company() ? $order->get_billing_company() : $order->get_formatted_billing_full_name(),
				'vatNumber' => $vat_number_clean,
				'email'     => $order->get_billing_email(),
				'address'   => array(
					'street'      => $order->get_billing_address_1(),
					'city'        => $order->get_billing_city(),
					'zipCode'     => $order->get_billing_postcode(),
					'countryCode' => $order->get_billing_country()
				)
			),
			'lines'          => $lines,
			'vatTotals'      => $vat_totals_api, 
			'paymentMethods' => $payment_methods,
			'attachments'    => $attachments,
			'totalInclVat'   => number_format( $calc_total_incl_vat, 2, '.', '' ),
			'totalVat'       => number_format( $calc_total_vat, 2, '.', '' ),
			'totalExclVat'   => number_format( $calc_total_excl, 2, '.', '' ),
			'payableRoundingAmount' => 0
		);

		// Send to Scrada
		if ( get_option( 'scrada_debug_mode' ) ) {
			error_log( 'Scrada JSON Payload: ' . json_encode( $payload ) );
		}
		
		$response = $this->call_scrada_api( $payload );

		// Handle Response
		if ( is_wp_error( $response ) ) {
			$order->add_order_note( 'Scrada Connection Error: ' . $response->get_error_message() );
		} else {
			$code = wp_remote_retrieve_response_code( $response );
			$body = wp_remote_retrieve_body( $response );
			
			if ( $code === 200 || $code === 201 ) {
				$order->update_meta_data( '_scrada_invoice_sent', 'yes' );
				$order->save();
				$order->add_order_note( 'Invoice successfully sent to Scrada (Peppol Network).' );
			} else {
				$order->add_order_note( 'Scrada API Failed (' . $code . '): ' . substr( $body, 0, 200 ) );
			}
		}
	}

	private function call_scrada_api( $payload ) {
		$test_mode  = get_option( 'scrada_test_mode' );
		$base_url   = $test_mode ? $this->api_test_url : $this->api_prod_url;
		$company_id = get_option( 'scrada_company_id' );
		
		$endpoint = "$base_url/company/$company_id/salesInvoice";

		$args = array(
			'body'    => json_encode( $payload ),
			'headers' => array(
				'Content-Type' => 'application/json',
				'X-API-KEY'    => get_option( 'scrada_api_key' ),
				'X-PASSWORD'   => get_option( 'scrada_api_secret' )
			),
			'timeout' => 45
		);

		return wp_remote_post( $endpoint, $args );
	}

	// Helper to fix rounding errors (e.g. 20.99% -> 21%)
	private function normalize_belgian_vat( $rate ) {
		// Belgium Rates: 0, 6, 12, 21
		$standard_rates = [0, 6, 12, 21];
		$closest = null;
		foreach ( $standard_rates as $std ) {
			if ( $closest === null || abs( $rate - $closest ) > abs( $rate - $std ) ) {
				$closest = $std;
			}
		}
		// If very close to a standard rate, snap to it. Otherwise keep calculated.
		return ( abs( $rate - $closest ) < 1 ) ? $closest : $rate;
	}

	// --- 3. AJAX VERIFICATION HANDLER ---
	public function ajax_verify_connection() {
		check_ajax_referer( 'scrada_verify_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( 'Unauthorized' );
		}

		$api_key    = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';
		$api_secret = isset( $_POST['api_secret'] ) ? sanitize_text_field( wp_unslash( $_POST['api_secret'] ) ) : '';
		$company_id = isset( $_POST['company_id'] ) ? sanitize_text_field( wp_unslash( $_POST['company_id'] ) ) : '';
		$test_mode  = isset( $_POST['test_mode'] ) ? intval( wp_unslash( $_POST['test_mode'] ) ) : 0;

		$base_url = $test_mode ? $this->api_test_url : $this->api_prod_url;
		
		// We try to list journals or just hit the company endpoint.
		// GET /company/{companyId}/journal is a safe bet to check read permissions.
		$endpoint = "$base_url/company/$company_id/journal";

		$args = array(
			'headers' => array(
				'Content-Type' => 'application/json',
				'X-API-KEY'    => $api_key,
				'X-PASSWORD'   => $api_secret
			),
			'timeout' => 15
		);

		$response = wp_remote_get( $endpoint, $args );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( 'Connection Failed: ' . $response->get_error_message() );
		}

		$code = wp_remote_retrieve_response_code( $response );
		
		if ( $code === 200 ) {
			wp_send_json_success( 'Connection Successful! Credentials valid.' );
		} elseif ( $code === 401 || $code === 403 ) {
			wp_send_json_error( 'Invalid Credentials (401/403)' );
		} elseif ( $code === 404 ) {
			wp_send_json_error( 'Company ID not found (404)' );
		} else {
			wp_send_json_error( 'Error ' . $code . ': ' . wp_remote_retrieve_response_message( $response ) );
		}
	}
	// --- LITE: MANUAL SEND META BOX ---
	public function add_scrada_meta_box() {
		add_meta_box(
			'scrada_manual_send',
			'Scrada E-Invoicing (Lite)',
			array( $this, 'render_scrada_meta_box' ),
			'shop_order',
			'side',
			'default'
		);
	}

	public function render_scrada_meta_box( $post ) {
		$order = wc_get_order( $post->ID );
		if ( ! $order ) return;

		$is_sent = $order->get_meta( '_scrada_invoice_sent' );
		
		echo '<div class="scrada-manual-send-container">';
		if ( $is_sent ) {
			echo '<p class="success-msg">✔ Invoice Sent to Scrada</p>';
			echo '<p><em>(Already processed)</em></p>';
		} else {
			// USAGE CHECK
			$usage_key = 'scrada_lite_usage_' . date( 'Y_m' );
			$usage_count = (int) get_option( $usage_key, 0 );
			$limit = 10;
			$remaining = $limit - $usage_count;

			if ( $remaining <= 0 ) {
				echo '<div class="notice notice-error inline" style="margin: 0 0 10px 0;"><p><strong>Monthly Limit Reached</strong></p></div>';
				echo '<p>You have sent 10/10 invoices this month.</p>';
				echo '<button type="button" class="button button-disabled" disabled>Limit Reached</button>';
				echo '<p style="margin-top: 10px;"><a href="#" target="_blank" class="button button-primary"><strong>Upgrade to Pro for Unlimited</strong></a></p>';
			} else {
				echo '<p>Manually send this invoice to Scrada.</p>';
				echo '<p class="remaining-msg">Invoices remaining this month: <strong>' . $remaining . '</strong></p>';
				
				// Nonce field for security
				wp_nonce_field( 'scrada_manual_send_action', 'scrada_manual_send_nonce' );
				
				echo '<button type="submit" name="scrada_manual_trigger" value="yes" class="button button-primary" style="width:100%;">Send Invoice Now</button>';
				echo '<p class="upgrade-link">Want automatic sending? <a href="#" target="_blank">Upgrade to Pro</a></p>';
			}
		}
		echo '</div>';
	}

	// --- LITE: USAGE COUNTER (Helper) ---
	private function check_and_increment_usage() {
		$usage_key = 'scrada_lite_usage_' . date( 'Y_m' );
		$usage_count = (int) get_option( $usage_key, 0 );
		$limit = 10;

		if ( $usage_count >= $limit ) {
			return false; // Limit reached
		}

		// Increment
		update_option( $usage_key, $usage_count + 1 );
		return true;
	}

	public function process_manual_send_to_scrada( $post_id ) {
		// Verify if our button was clicked
		if ( ! isset( $_POST['scrada_manual_trigger'] ) || $_POST['scrada_manual_trigger'] !== 'yes' ) {
			return;
		}

		// Security Checks
		if ( ! isset( $_POST['scrada_manual_send_nonce'] ) || ! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['scrada_manual_send_nonce'] ) ), 'scrada_manual_send_action' ) ) {
			return;
		}

		if ( ! current_user_can( 'edit_shop_order', $post_id ) ) {
			return;
		}

		// Run the sending logic
		$this->send_invoice_to_scrada( $post_id );
	}
}

new E_Invoicing_Connector_For_Scrada_Lite();
